import express from 'express';
import cors from 'cors';
import rateLimit from 'express-rate-limit';
import fs from 'node:fs';
import path from 'node:path';

const {
    PORT = 3000,
    INTAKE_SECRET="test",           // required — match the header from the browser
    LOG_DIR = './data/logs', // local NDJSON storage
    LOG_FILE_PREFIX = 'client-errors',
} = process.env;

if (!INTAKE_SECRET) {
    console.error('Missing INTAKE_SECRET env var');
    process.exit(1);
}

const app = express();
app.use(cors({ origin: ALLOW_ORIGIN === '*' ? true : ALLOW_ORIGIN.split(',') }));
app.set('trust proxy', 1);
app.use(rateLimit({ windowMs: 60_000, max: 300, standardHeaders: true, legacyHeaders: false }));
fs.mkdirSync(LOG_DIR, { recursive: true });

function currentLogPath() {
    const day = new Date().toISOString().slice(0, 10); // YYYY-MM-DD
    return path.join(LOG_DIR, `${LOG_FILE_PREFIX}-${day}.ndjson`);
}
function appendNdjson(obj) {
    fs.appendFile(currentLogPath(), JSON.stringify(obj) + '\n', (err) => {
        if (err) console.error('Log write error', err);
    });
}

app.post('/intake/json', express.json({ limit: '200kb' }), (req, res) => {
    if (req.get('x-intake-secret') !== INTAKE_SECRET) return res.sendStatus(401);

    const { app: appName, timestamp, event } = req.body || {};
    if (!event) return res.status(400).json({ error: 'Missing event' });

    const logEvent = {
        type: 'sentry-like',
        receivedAt: Date.now(),
        clientSentAt: timestamp || null,
        app: appName || 'angular',
        ip: req.ip,
        ua: req.get('user-agent'),
        event, // keep original Sentry SDK event intact for later analysis
    };

    appendNdjson(logEvent);
    res.sendStatus(202);
});

app.get('/healthz', (req, res) => res.json({ ok: true }));

app.listen(PORT, () => console.log(`No-Sentry intake listening on :${PORT}`));